/* ------------------------------------------------------
 * Copyright (c) 2023 udon
 * Released under the MIT license
 * http://opensource.org/licenses/mit-license.php
 * ------------------------------------------------------*/
/*:
@target MZ
@author udon
@plugindesc
現在ターン表示プラグイン ver.1.0.0

@help
戦闘中、現在のターン数を閲覧するためのウィンドウを表示します。

## ツクールバージョン ##
RPGツクールMZ Version 1.6.1 で動作確認。

## 最終更新日 ##
2023/06/18

## 備考 ##
不具合報告や要望は以下まで連絡いただけると助かります。
https://twitter.com/udon2481

@param WindowConfig
@text ターンウィンドウの設定
@desc ターンウィンドウ自体への設定内容を指定できます。

    @param WindowLocationX
    @type number
    @text X座標
    @desc ターンウィンドウの表示位置(X座標)を指定します。(デフォルト値:712)
    @default 712
    @parent WindowConfig

    @param WindowLocationY
    @type number
    @text Y座標
    @desc ターンウィンドウの表示位置(Y座標)を指定します。(デフォルト値:0)
    @default 0
    @parent WindowConfig

    @param WindowSizeWidth
    @type number
    @text 横幅
    @desc ターンウィンドウの横幅を指定します。(デフォルト値:96)
    @default 96
    @parent WindowConfig

    @param WindowSizeHeight
    @type number
    @text 高さ
    @desc ターンウィンドウの高さを指定します。(デフォルト値:96)
    @default 96
    @parent WindowConfig

    @param SpriteFileName
    @type file
    @text 背景画像
    @desc ターンウィンドウの代わりに表示する画像ファイルを指定します。
    @dir img/system/
    @parent WindowConfig

@param TurnConfig
@text 表示内容の設定
@desc ターンウィンドウ内の表示内容を指定できます。

    @param TurnNameConfig
    @text ターン名称
    @desc ターンウィンドウ内に数字とともに表示する名称を表します。
    @parent TurnConfig

        @param TurnName
        @type string
        @text 名称
        @desc 名称を指定します。表示したくない場合は未入力にしてください。(デフォルト値:Turn)
        @default Turn
        @parent TurnNameConfig

        @param TurnNameFontSize
        @type number
        @text 文字サイズ
        @desc ターン名称の文字サイズを指定します。(デフォルト値:26)
        @default 26
        @parent TurnNameConfig

        @param TurnNameLocationX
        @type number
        @text X座標
        @desc ターンウィンドウ内における名称の表示位置(X座標)を指定します。(デフォルト値:8)
        @default 8
        @min -8
        @parent TurnNameConfig

        @param TurnNameLocationY
        @type number
        @text Y座標
        @desc ターンウィンドウ内における名称の表示位置(Y座標)を指定します。(デフォルト値:0)
        @default 0
        @min -8
        @parent TurnNameConfig

        @param TurnNameAlign
        @type combo
        @text 揃え位置
        @desc 名称の揃え位置(text-align)を指定します。(デフォルト値:left)
        @option left
        @option center
        @option right
        @default left
        @parent TurnNameConfig

    @param TurnNumberConfig
    @text ターン数値
    @desc 現在の戦闘におけるターンを表します。
          これは、1からはじまる自然数です。
    @parent TurnConfig

        @param TurnNumberFontSize
        @type number
        @text 文字サイズ
        @desc ターン数値の文字サイズを指定します。(デフォルト値:26)
        @default 26
        @parent TurnNumberConfig

        @param TurnNumberLocationX
        @type number
        @text X座標
        @desc ターンウィンドウ内におけるターンの数値の表示位置(X座標)を指定します。(デフォルト値:8)
        @default 8
        @min -8
        @parent TurnNumberConfig

        @param TurnNumberLocationY
        @type number
        @text Y座標
        @desc ターンウィンドウ内におけるターンの数値の表示位置(Y座標)を指定します。(デフォルト値:38)
        @default 38
        @min -8
        @parent TurnNumberConfig

        @param TurnNumberAlign
        @type combo
        @text 揃え位置
        @desc ターンの数値の揃え位置(text-align)を指定します。(デフォルト値:center)
        @option left
        @option center
        @option right
        @default center
        @parent TurnNumberConfig
*/

var Imported = Imported || {};
Imported["VblTurnCounterWindow"] = 1.0;

(() => {
    "use strict";

    const pluginName = "VblTurnCounterWindow";
    const params = PluginManager.parameters(pluginName);

    class VblTurnCounterWindowArguments { }
    // ウィンドウの設定
    VblTurnCounterWindowArguments.windowLocationX = Number(params.WindowLocationX || 712);
    VblTurnCounterWindowArguments.windowLocationY = Number(params.WindowLocationY || 0);
    VblTurnCounterWindowArguments.windowWidth = Number(params.WindowSizeWidth || 96);
    VblTurnCounterWindowArguments.windowHeight = Number(params.WindowSizeHeight || 96);
    // ウィンドウの背景画像の設定
    VblTurnCounterWindowArguments.spriteFileName = params.SpriteFileName;
    VblTurnCounterWindowArguments.isUseSprite = (params.SpriteFileName !== "");
    // ターン名称の設定
    VblTurnCounterWindowArguments.turnName = params.TurnName;
    VblTurnCounterWindowArguments.turnNameFontSize = Number(params.TurnNameFontSize || 26);
    VblTurnCounterWindowArguments.turnNameLocationX = Number(params.TurnNameLocationX || 8);
    VblTurnCounterWindowArguments.turnNameLocationY = Number(params.TurnNameLocationY || 0);
    VblTurnCounterWindowArguments.turnNameAlign = params.TurnNameAlign || "left";
    // ターン数の設定
    VblTurnCounterWindowArguments.turnNumberFontSize = Number(params.TurnNumberFontSize || 26);
    VblTurnCounterWindowArguments.turnNumberLocationX = Number(params.TurnNumberLocationX || 8);
    VblTurnCounterWindowArguments.turnNumberLocationY = Number(params.TurnNumberLocationY || 38);
    VblTurnCounterWindowArguments.turnNumberAlign = params.TurnNumberAlign || "center";

    // プラグインで使用する定数
    class VblTurnCounterWindowConst { }
    // ウィンドウの枠線の幅
    VblTurnCounterWindowConst.windowBorderWidth = 8;
    // ウィンドウ端の描画不可能な範囲の幅
    // 通常のウィンドウを使用する場合、枠線の内側にさらに描画不可能な領域が存在するようだ。その領域分の補正を行う。
    VblTurnCounterWindowConst.undrawableWidth = VblTurnCounterWindowConst.windowBorderWidth + 7;

    // 背景画像を使用しないターンウィンドウ
    class VblTurnCounterWindow extends Window_Base {
        constructor(rect) {
            super(rect);

            this.contents.fontSize = 1;
            this.openness = 0;
            this.opacity = this.windowOpacity();
            this.refresh();
        }

        static create() {
            if (VblTurnCounterWindowArguments.isUseSprite) {
                const rect = VblTurnCounterSprite.windowRectangle();
                return new VblTurnCounterSprite(rect);
            } else {
                const rect = VblTurnCounterWindow.windowRectangle();
                return new VblTurnCounterWindow(rect);
            }
        }

        static windowRectangle() {
            const x = VblTurnCounterWindowArguments.windowLocationX;
            const y = VblTurnCounterWindowArguments.windowLocationY;
            const width = VblTurnCounterWindowArguments.windowWidth;
            const height = VblTurnCounterWindowArguments.windowHeight;
            return new Rectangle(x, y, width, height);
        }

        drawTurnName() {
            this.contents.fontSize = VblTurnCounterWindowArguments.turnNameFontSize;

            const name = VblTurnCounterWindowArguments.turnName;
            const conf = this.turnNameConfig();
            this.drawText(name, conf.x, conf.y, conf.width, conf.align);
        }

        drawTurnNumber() {
            this.contents.fontSize = VblTurnCounterWindowArguments.turnNumberFontSize;

            const turn = String(BattleManager.turnCountForVblTurnCounterWindow());
            const conf = this.turnNumberConfig();
            this.drawText(turn, conf.x, conf.y, conf.width, conf.align);
        }

        open() {
            super.open();
            this.refresh();
        }

        refresh() {
            this.contents.clear();

            this.drawTurnName();
            this.drawTurnNumber();
        }

        turnNameConfig() {
            return {
                x: VblTurnCounterWindowArguments.turnNameLocationX,
                y: VblTurnCounterWindowArguments.turnNameLocationY,
                width: VblTurnCounterWindowArguments.windowWidth - VblTurnCounterWindowArguments.turnNameLocationX - VblTurnCounterWindowConst.undrawableWidth * 2,
                align: VblTurnCounterWindowArguments.turnNameAlign
            };
        }

        turnNumberConfig() {
            return {
                x: VblTurnCounterWindowArguments.turnNumberLocationX,
                y: VblTurnCounterWindowArguments.turnNumberLocationY,
                width: VblTurnCounterWindowArguments.windowWidth - VblTurnCounterWindowArguments.turnNumberLocationX - VblTurnCounterWindowConst.undrawableWidth * 2,
                align: VblTurnCounterWindowArguments.turnNumberAlign
            };
        }

        windowOpacity() {
            return 255;
        }
    }

    // 背景画像を使用するターンウィンドウ
    class VblTurnCounterSprite extends VblTurnCounterWindow {
        static windowRectangle() {
            const x = VblTurnCounterWindowArguments.windowLocationX - VblTurnCounterWindowConst.windowBorderWidth;
            const y = VblTurnCounterWindowArguments.windowLocationY - VblTurnCounterWindowConst.windowBorderWidth;
            const width = VblTurnCounterWindowArguments.windowWidth + VblTurnCounterWindowConst.windowBorderWidth * 2;
            const height = VblTurnCounterWindowArguments.windowHeight + VblTurnCounterWindowConst.windowBorderWidth * 2;
            return new Rectangle(x, y, width, height);
        }

        refresh() {
            this.contents.clear();

            this.drawSprite();
            this.drawTurnName();
            this.drawTurnNumber();
        }

        drawSprite() {
            const imgName = VblTurnCounterWindowArguments.spriteFileName;
            const bitmap = ImageManager.loadSystem(imgName);
            this.contents.blt(bitmap, 0, 0, bitmap.width, bitmap.height, 0, 0);
        }

        turnNameConfig() {
            return {
                x: VblTurnCounterWindowArguments.turnNameLocationX + VblTurnCounterWindowConst.windowBorderWidth,
                y: VblTurnCounterWindowArguments.turnNameLocationY + VblTurnCounterWindowConst.windowBorderWidth,
                width: VblTurnCounterWindowArguments.windowWidth - VblTurnCounterWindowArguments.turnNameLocationX - VblTurnCounterWindowConst.undrawableWidth * 2,
                align: VblTurnCounterWindowArguments.turnNameAlign
            };
        }

        turnNumberConfig() {
            return {
                x: VblTurnCounterWindowArguments.turnNumberLocationX + VblTurnCounterWindowConst.windowBorderWidth,
                y: VblTurnCounterWindowArguments.turnNumberLocationY + VblTurnCounterWindowConst.windowBorderWidth,
                width: VblTurnCounterWindowArguments.windowWidth - VblTurnCounterWindowArguments.turnNumberLocationX - VblTurnCounterWindowConst.undrawableWidth * 2,
                align: VblTurnCounterWindowArguments.turnNumberAlign
            };
        }

        windowOpacity() {
            return 0;
        }
    }

    // メソッドのフック
    // まず、baseの処理を実行したあと、書き換えた処理を実行する
    function postfaceHook(baseClass, methodName, newAction) {
        if (baseClass.prototype[methodName]) {
            baseClass = baseClass.prototype;
        }
        else if (!baseClass[methodName]) {
            throw new Error("hook destination is not found.");
        }

        const oldAction = baseClass[methodName];
        baseClass[methodName] = function () {
            const returnValue = oldAction.apply(this, arguments);

            // 元の処理の実行結果を引数に追加する
            arguments[arguments.length] = returnValue;
            arguments.length++;

            return newAction.apply(this, arguments);
        };
    }

    postfaceHook(Scene_Battle, "commandSkill", function () {
        this._turnWindow.hide();
    });

    postfaceHook(Scene_Battle, "commandItem", function () {
        this._turnWindow.hide();
    });

    postfaceHook(Scene_Battle, "createAllWindows", function () {
        this.createVblTurnCounterWindow();
    });

    postfaceHook(Scene_Battle, "onActorCancel", function () {
        switch (this._actorCommandWindow.currentSymbol()) {
            case "skill":
            case "item":
                this._turnWindow.hide();
                break;
        }
    });

    postfaceHook(Scene_Battle, "onEnemyCancel", function () {
        switch (this._actorCommandWindow.currentSymbol()) {
            case "skill":
            case "item":
                this._turnWindow.hide();
                break;
            default:
                this._turnWindow.show();
        }
    });

    postfaceHook(Scene_Battle, "onSkillOk", function () {
        this._turnWindow.show();
    });

    postfaceHook(Scene_Battle, "onSkillCancel", function () {
        this._turnWindow.show();
    });

    postfaceHook(Scene_Battle, "onItemOk", function () {
        this._turnWindow.show();
    });

    postfaceHook(Scene_Battle, "onItemCancel", function () {
        this._turnWindow.show();
    });

    postfaceHook(Scene_Battle, "update", function () {
        if (BattleManager.isInputting() && this._turnWindow.isClosed()) {
            this._turnWindow.open();
        };
        if (BattleManager.isInTurn() && this._turnWindow.isClosed()) {
            this._turnWindow.open();
        };
        if (BattleManager.isAborting() || BattleManager.isBattleEnd()) {
            this._turnWindow.close();
        };
        if (this._turnWindow._turnCount !== BattleManager.turnCountForVblTurnCounterWindow()) {
            this._turnWindow.refresh();
        };
    });

    Scene_Battle.prototype.createVblTurnCounterWindow = function () {
        this._turnWindow = VblTurnCounterWindow.create();
        this.addWindow(this._turnWindow);
        this._turnWindow.refresh();
    };

    BattleManager.turnCountForVblTurnCounterWindow = function () {
        if (this.isInputting()) {
            return $gameTroop._turnCount + 1;
        }
        else {
            return $gameTroop._turnCount;
        }
    };
})();
